#!/bin/bash

# Initialize a variable with the default Helm executable
HELM_CMD="helm"

# Check if Helm 3 is installed and update the variable if it is
if command -v helm3 &> /dev/null
then
    echo "Helm3 is installed, using helm3."
    HELM_CMD="helm3"
else
    echo "Using default Helm."
fi

# Remove any existing package files + create a Chart copy
rm -f runtime*.tgz
CHART_FILE="Chart.yaml"
cp "${CHART_FILE}" "${CHART_FILE}.tmp"

# User params
SENSORS_VERSION=$1
PUSH_CHART=$2
ARTIFACTORY_REPO=$3

function cleanup {
    # Remove the backup file
    echo "Cleaning up..."
    cp "${CHART_FILE}.tmp" "${CHART_FILE}"
    rm "${CHART_FILE}.tmp"
    rm "${CHART_FILE}.${BACKUP_SUFFIX}"
}

# Check if service version provided
if [ "$#" -lt 1 ]; then
    echo "Usage: $0 <sensors-version> [--push] [entplus-user] [entplus-token]"
    echo -e "\t service_version - Version of the service in the format x.y.z. For example, 0.24.1"
    echo -e "\t --push - Optional argument. If provided, the Helm chart will be pushed to Artifactory and the backup file will be deleted"
    echo -e "\t entplus-user - The username for Entplus. Required if --push is used. For example, yahavo"
    echo -e "\t entplus-token - The Identity/API token for Entplus. Required if --push is used."
    cleanup
    exit 0
fi

# Package location + version
CHART_VERSION="10${SENSORS_VERSION}"
BACKUP_SUFFIX="bak.yaml"
BACK_TEMP_CHART_FILE="${CHART_FILE}.${BACKUP_SUFFIX}"

# Check if Chart.yaml file exists
if [ ! -f "$CHART_FILE" ]; then
    echo "Chart.yaml file not found in the ./runtime directory"
    exit 2
fi

# Update Chart.yaml with the new version + appVersion
sed -i.${BACKUP_SUFFIX} "s/^appVersion: .*/appVersion: \"${SENSORS_VERSION}\"/" "${CHART_FILE}"
sed -i.${BACKUP_SUFFIX} "s/^version: .*/version: ${CHART_VERSION}/" "${CHART_FILE}"

# Package the Helm chart
$HELM_CMD package . || { echo 'Helm package failed' ; cleanup ; exit 1; }

# Run helm lint on the package
PACKAGE_LOCATION=$(find . -name "*${CHART_VERSION}.tgz")
$HELM_CMD lint ${PACKAGE_LOCATION} || { echo 'Helm lint failed' ; cleanup ; exit 2; }

# Remove the backup file + Copy the Chart file back
cp "${CHART_FILE}.tmp" "${CHART_FILE}"

# Check if linting succeeded
if [ $? -eq 0 ]; then
    echo "Package created successfully!"
    if [ "$PUSH_CHART" == "--push" ]; then
        # Publish the package to Artifactory
        echo "Publishing the Helm Chart to Artifactory"
        VERSION=runtime-sensors-${CHART_VERSION}.tgz
        jfrog rt upload --fail-no-op "${VERSION}" "$ARTIFACTORY_REPO/$VERSION"
    fi
    cleanup
else
    cleanup
    echo "Error linting package"
    exit 4
fi
